/*
 * Decompiled with CFR 0.152.
 */
package de.governikus.csl.tsp;

import java.io.IOException;
import java.security.Security;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.cmp.PKIFailureInfo;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.x509.ExtendedKeyUsage;
import org.bouncycastle.asn1.x509.Extensions;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cms.SignerId;
import org.bouncycastle.cms.jcajce.JcaSimpleSignerInfoVerifierBuilder;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.tsp.TSPException;
import org.bouncycastle.tsp.TSPValidationException;
import org.bouncycastle.tsp.TimeStampRequest;
import org.bouncycastle.tsp.TimeStampResponse;
import org.bouncycastle.tsp.TimeStampToken;
import org.bouncycastle.tsp.TimeStampTokenInfo;
import org.bouncycastle.util.Selector;
import org.bouncycastle.util.Store;
import org.bouncycastle.util.StoreException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TimestampResponseValidator {
    private static final String MSG_NO_TOKEN_PRESENT = "no token present";
    private static final String MSG_TIMESTAMP_TOKEN_INFO_NULL = "no timestamp token info present at timestamp token";
    public static final String OID_EXTENSION_TIMESTAMPTOKEN_QC_STATEMENTS = "1.3.6.1.5.5.7.1.3";
    public static final int MAX_DELTA_TIMESTAMP_GENERATION_TIME_ERROR = 3600000;
    public static final int MAX_DELTA_TIMESTAMP_GENERATION_TIME_WARNING = 60000;
    private static final Logger LOGGER = LoggerFactory.getLogger(TimestampResponseValidator.class);

    private TimestampResponseValidator() {
    }

    public static boolean validateTimestampResponse(byte[] timestampResponseBytes, TimeStampRequest timestampRequest, TimestampTokenSignerCertificateValidator timestampTokenSignerCertificateValidator, Date date) {
        if (timestampResponseBytes == null || timestampResponseBytes.length == 0) {
            return false;
        }
        try {
            return TimestampResponseValidator.validateTimestampResponse(new TimeStampResponse(timestampResponseBytes), timestampRequest, timestampTokenSignerCertificateValidator, date);
        }
        catch (IOException | TSPException e) {
            LOGGER.error("malformed TimestampResponse", e);
            return false;
        }
    }

    public static boolean validateTimestampResponse(TimeStampResponse timestampResponse, TimeStampRequest timestampRequest, TimestampTokenSignerCertificateValidator timestampTokenSignerCertificateValidator, Date date) {
        if (!TimestampResponseValidator.validateTimestampResponseStatus(timestampResponse)) {
            return false;
        }
        TimeStampToken timestampToken = timestampResponse.getTimeStampToken();
        if (!TimestampResponseValidator.validateTimestampTokenExists(timestampToken)) {
            return false;
        }
        if (!TimestampResponseValidator.validateTimestampTokenMatchesETSIRequirements(timestampToken)) {
            return false;
        }
        if (timestampRequest != null) {
            try {
                timestampResponse.validate(timestampRequest);
            }
            catch (TSPException e) {
                LOGGER.info("Validation of the timestamp response failed.", (Throwable)e);
                return false;
            }
        }
        if (!TimestampResponseValidator.validateTimestampTokenMatchesRequestRequirements(timestampToken, timestampRequest)) {
            return false;
        }
        if (!TimestampResponseValidator.validateTimestampTokenGenerationTime(timestampToken, date)) {
            return false;
        }
        return TimestampResponseValidator.validateTimestampTokenSignature(timestampToken, timestampTokenSignerCertificateValidator, date);
    }

    public static boolean validateTimestampTokenGenerationTime(TimeStampToken timestampToken, Date date) {
        Date genTime;
        long delta;
        if (timestampToken == null) {
            LOGGER.error(MSG_NO_TOKEN_PRESENT);
            return false;
        }
        Date checkDate = date;
        if (checkDate == null) {
            LOGGER.debug("using current time");
            checkDate = new Date(System.currentTimeMillis());
        }
        if ((delta = Math.abs((genTime = timestampToken.getTimeStampInfo().getGenTime()).getTime() - checkDate.getTime())) >= 3600000L) {
            LOGGER.error("current system time seems to be misconfigured, please check: {}", (Object)new Date(System.currentTimeMillis()));
            return false;
        }
        if (delta >= 60000L) {
            LOGGER.warn("current system time seems to be misconfigured, please check: {}", (Object)new Date(System.currentTimeMillis()));
            return true;
        }
        return true;
    }

    public static boolean validateTimestampTokenMatchesETSIRequirements(TimeStampToken timestampToken) {
        if (timestampToken == null) {
            LOGGER.error(MSG_NO_TOKEN_PRESENT);
            return false;
        }
        if (timestampToken.getSignedAttributes().get(PKCSObjectIdentifiers.id_aa_signingCertificate) == null && timestampToken.getSignedAttributes().get(PKCSObjectIdentifiers.id_aa_signingCertificateV2) == null) {
            LOGGER.error("no signing certificate or signing certificateV2 present");
            return false;
        }
        return true;
    }

    public static boolean validateTimestampTokenMatchesRequestRequirements(TimeStampToken timestampToken, TimeStampRequest timestampRequest) {
        if (timestampToken == null) {
            LOGGER.error(MSG_NO_TOKEN_PRESENT);
            return false;
        }
        return TimestampResponseValidator.validateTimestampTokenInfoMatchesRequestRequirements(timestampToken.getTimeStampInfo(), timestampRequest);
    }

    public static boolean validateTimestampTokenInfoMatchesETSIRequirements(TimeStampTokenInfo timestampTokenInfo) {
        if (timestampTokenInfo == null) {
            LOGGER.debug(MSG_TIMESTAMP_TOKEN_INFO_NULL);
            return false;
        }
        if (timestampTokenInfo.getExtensions() == null || timestampTokenInfo.getExtensions().getExtension(new ASN1ObjectIdentifier(OID_EXTENSION_TIMESTAMPTOKEN_QC_STATEMENTS)) == null) {
            LOGGER.warn("TimeStampToken does not contain extension of qualified Trust Service Providers issuing Time-Stamps, missing exension: 1.3.6.1.5.5.7.1.3");
        }
        return true;
    }

    public static boolean validateTimestampTokenInfoMatchesRequestRequirements(TimeStampTokenInfo timestampTokenInfo, TimeStampRequest timestampRequest) {
        if (!TimestampResponseValidator.validateTimestampTokenInfoMatchesETSIRequirements(timestampTokenInfo)) {
            return false;
        }
        if (timestampRequest == null) {
            LOGGER.warn("timestamp token can not be checked against timestamp request, request not specified");
            return true;
        }
        if (timestampTokenInfo.getNonce() == null && timestampRequest.getNonce() == null) {
            LOGGER.debug("no nonce present");
        } else {
            if (timestampTokenInfo.getNonce() != null && !timestampTokenInfo.getNonce().equals(timestampRequest.getNonce())) {
                LOGGER.debug("nonce of timestamp token does not match nonce of timestamp request");
                return false;
            }
            if (timestampTokenInfo.getNonce() == null && timestampRequest.getNonce() != null) {
                LOGGER.debug("nonce not validated, nonce not present at timestamp token info, present at request");
            }
        }
        if (timestampRequest.getReqPolicy() != null && !timestampRequest.getReqPolicy().equals((ASN1Primitive)timestampTokenInfo.getPolicy())) {
            LOGGER.debug("timestamp token policy does not match policy timestamp request");
            return false;
        }
        if (!timestampTokenInfo.getMessageImprintAlgOID().equals((ASN1Primitive)timestampRequest.getMessageImprintAlgOID())) {
            LOGGER.debug("message imprint digest algorithm OID of timestamp token does not match message imprint digest algorithm OID of timestamp request");
            return false;
        }
        if (!TimestampResponseValidator.validateTimestampTokenInfoDigestValue(timestampTokenInfo, timestampRequest.getMessageImprintDigest())) {
            LOGGER.debug("message imprint digest of timestamp token does not match message imprint digest of timestamp request");
            return false;
        }
        return true;
    }

    public static boolean validateTimestampTokenInfoDigestValue(TimeStampTokenInfo timestampTokenInfo, byte[] expectedDigest) {
        if (timestampTokenInfo == null) {
            LOGGER.debug(MSG_TIMESTAMP_TOKEN_INFO_NULL);
            return false;
        }
        if (expectedDigest == null || expectedDigest.length == 0) {
            LOGGER.debug("expected digest is null or empty, can not be validated");
            return false;
        }
        if (!Arrays.equals(timestampTokenInfo.getMessageImprintDigest(), expectedDigest)) {
            LOGGER.debug("message imprint digest of timestamp token does not match expected digest value");
            return false;
        }
        return true;
    }

    public static boolean validateTimestampTokenExists(TimeStampToken timestampToken) {
        if (timestampToken == null) {
            LOGGER.error("no token present, token null at granted response");
            return false;
        }
        return true;
    }

    public static boolean validateTimestampResponseStatus(TimeStampResponse timestampResponse) {
        if (timestampResponse == null) {
            LOGGER.error("no timestamp token response present");
            return false;
        }
        if (0 != timestampResponse.getStatus()) {
            TimestampResponseValidator.logTimestampResponseStatus(timestampResponse);
            LOGGER.error("timestamp token response indicates failure status");
            return false;
        }
        if (timestampResponse.getFailInfo() != null) {
            LOGGER.error("timestamp token response failure info present - GRANTED");
            return false;
        }
        return true;
    }

    public static boolean validateTimestampTokenSignature(TimeStampToken timestampToken, TimestampTokenSignerCertificateValidator timestampTokenSignerCertificateValidator, Date date) {
        block16: {
            if (timestampToken == null) {
                LOGGER.debug("no timestamp token present");
                return false;
            }
            Store tspCertificatesStore = timestampToken.getCertificates();
            Date genTime = timestampToken.getTimeStampInfo().getGenTime();
            if (tspCertificatesStore != null) {
                try {
                    SignerId timestampTokenSignerId = timestampToken.getSID();
                    if (timestampTokenSignerId == null) {
                        LOGGER.debug("no timestamp token signer id is present");
                        return false;
                    }
                    Collection timestampTokenSignerCertificates = tspCertificatesStore.getMatches((Selector)timestampTokenSignerId);
                    if (!timestampTokenSignerCertificates.isEmpty()) {
                        if (timestampTokenSignerCertificates.size() != 1) {
                            LOGGER.info("Invalid timestamp: expected exactly one certificate for each signer in timestamp - external TSP creation does not work correctly");
                            return false;
                        }
                        Object timestampTokenSignerCertificateObject = timestampTokenSignerCertificates.iterator().next();
                        if (timestampTokenSignerCertificateObject instanceof X509CertificateHolder) {
                            X509CertificateHolder timestampTokenSignerCertificate = (X509CertificateHolder)timestampTokenSignerCertificateObject;
                            if (!timestampTokenSignerId.getIssuer().equals((Object)timestampTokenSignerCertificate.getIssuer())) {
                                LOGGER.info("Invalid timestamp: certificate issuer not as expected.");
                                return false;
                            }
                            if (!timestampTokenSignerCertificate.isValidOn(genTime)) {
                                LOGGER.info("Invalid timestamp: certificate not valid on generation time.");
                                return false;
                            }
                            if (timestampTokenSignerCertificateValidator != null && !timestampTokenSignerCertificateValidator.validate(timestampTokenSignerCertificate, genTime)) {
                                LOGGER.debug("signer certificate not valid");
                                return false;
                            }
                            if (timestampTokenSignerCertificateValidator == null) {
                                LOGGER.warn("timestamp token signer certificate not checked, no validator specified");
                            } else {
                                LOGGER.debug("signer certificate is valid");
                            }
                            return TimestampResponseValidator.validateTimestampTokenSignatureSimple(timestampToken, timestampTokenSignerCertificate);
                        }
                        break block16;
                    }
                    LOGGER.warn("timestamp token does not contain timestamp token signer certificate - check internal/external timestamp token creation");
                    return false;
                }
                catch (StoreException e) {
                    LOGGER.warn("error with certificates contained by timestamp token: ", (Throwable)e);
                }
                catch (CertificateException e) {
                    LOGGER.warn("Certificate used for timestamp token: ", (Throwable)e);
                }
                catch (Exception e) {
                    LOGGER.warn("failed to validate timestamp token, provider registered: 'BC'");
                }
            } else {
                LOGGER.warn("no certificate present at timestamp token");
            }
        }
        return false;
    }

    public static boolean validateTimestampTokenSignatureSimple(TimeStampToken timestampToken, X509CertificateHolder timestampTokenSignerCertificate) throws OperatorCreationException, CertificateException {
        if (timestampToken == null) {
            LOGGER.debug("no timestamp token present");
            return false;
        }
        if (timestampTokenSignerCertificate == null) {
            LOGGER.debug("no timestamp token signer certificate present");
            return false;
        }
        if (Security.getProvider("BC") == null) {
            LOGGER.debug("required BouncyCastleProvider currently not registered, signature validation is not possible");
            return false;
        }
        try {
            timestampToken.validate(new JcaSimpleSignerInfoVerifierBuilder().setProvider("BC").build(timestampTokenSignerCertificate));
            LOGGER.debug("timestamp token validated successfully (certificate of timestamp token is used)");
            return true;
        }
        catch (TSPValidationException e) {
            LOGGER.warn("timestamp token signature is invalid: " + e.getMessage(), (Throwable)e);
            return false;
        }
        catch (TSPException e) {
            LOGGER.warn("timestamp token validation failed internally: " + e.getMessage(), (Throwable)e);
            return false;
        }
    }

    public static void logTimestampResponseStatus(TimeStampResponse timeStampResponse) {
        if (timeStampResponse == null) {
            LOGGER.error("no timestamp response to log status for");
            return;
        }
        int status = timeStampResponse.getStatus();
        if (0 == status) {
            return;
        }
        if (LOGGER.isInfoEnabled()) {
            TimestampResponseValidator.logPKIStatus(timeStampResponse, status);
            TimestampResponseValidator.logPKIFailureInfo(timeStampResponse);
        }
    }

    private static void logPKIFailureInfo(TimeStampResponse timeStampResponse) {
        PKIFailureInfo failInfo = timeStampResponse.getFailInfo();
        if (LOGGER.isInfoEnabled() && failInfo != null) {
            boolean[] b = TimestampResponseValidator.toBooleanArray((DERBitString)failInfo);
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < b.length; ++i) {
                if (!b[i]) continue;
                sb.append(i);
                if (i == b.length - 1) continue;
                sb.append(", ");
            }
            LOGGER.info("timestamp response failure status info: {}\nPKIFailureInfo ::= BIT STRING {\n badAlg               (0), -- unrecognized or unsupported Algorithm Identifier\n badMessageCheck      (1), -- integrity check failed (e.g., signature did not verify)\n badRequest           (2), -- transaction not permitted or supported\n badTime              (3), -- messageTime was not sufficiently close to the system time, as defined by local policy\n badCertId            (4), -- no certificate could be found matching the provided criteria\n badDataFormat        (5), -- the data submitted has the wrong format\n wrongAuthority       (6), -- the authority indicated in the request is different from the one creating the response token\n incorrectData        (7), -- the requester's data is incorrect (for notary services)\n missingTimeStamp     (8), -- when the timestamp is missing but should be there (by policy)\n badPOP               (9)  -- the proof-of-possession failed\n certRevoked         (10)\n certConfirmed       (11)\n wrongIntegrity      (12)\n badRecipientNonce   (13)\n timeNotAvailable    (14), -- the TSA's time source is not available\n unacceptedPolicy    (15), -- the requested TSA policy is not supported by the TSA\n unacceptedExtension (16), -- the requested extension is not supported by the TSA\n addInfoNotAvailable (17), -- the additional information requested could not be understood -- or is not available\n badSenderNonce      (18)\n badCertTemplate     (19)\n signerNotTrusted    (20)\n transactionIdInUse  (21)\n unsupportedVersion  (22)\n notAuthorized       (23)\n systemUnavail       (24)\n systemFailure       (25), -- the request cannot be handled due to system failure\n duplicateCertReq    (26)}", (Object)sb.toString());
        }
    }

    private static void logPKIStatus(TimeStampResponse timeStampResponse, int status) {
        if (LOGGER.isInfoEnabled()) {
            StringBuilder pkiStatus = new StringBuilder("timestamp response status: " + status + " - " + timeStampResponse.getStatusString() + "\n");
            pkiStatus.append("(0) PKIStatus.GRANTED\n(1) PKIStatus.GRANTED_WITH_MODS\n(2) PKIStatus.REJECTION\n(3) PKIStatus.WAITING\n(4) PKIStatus.REVOCATION_WARNING\n(5) PKIStatus.KEY_UPDATE_WARNING\n(6) PKIStatus.KEY_UPDATE_WARNING");
            LOGGER.info(pkiStatus.toString());
        }
    }

    public static boolean[] toBooleanArray(DERBitString asn1bitString) {
        if (asn1bitString != null) {
            byte[] bytes = asn1bitString.getBytes();
            boolean[] boolId = new boolean[bytes.length * 8 - asn1bitString.getPadBits()];
            for (int i = 0; i != boolId.length; ++i) {
                boolId[i] = (bytes[i / 8] & 0xFF & 128 >>> i % 8) != 0;
            }
            return boolId;
        }
        return new boolean[0];
    }

    public static boolean checkTS(TimeStampResponse timestampResponse, TimeStampRequest timestampRequest) {
        return TimestampResponseValidator.validateTimestampResponse(timestampResponse, timestampRequest, null, null);
    }

    public static interface TimestampTokenSignerCertificateValidator {
        default public boolean validate(X509Certificate certificate, Date date) {
            if (certificate == null) {
                return false;
            }
            try {
                return this.validate(new X509CertificateHolder(certificate.getEncoded()), date);
            }
            catch (IOException | CertificateEncodingException e) {
                LOGGER.error("certificate encoding failure detected: ", (Throwable)e);
                return false;
            }
        }

        default public boolean validate(X509CertificateHolder certificateHolder, Date date) {
            if (certificateHolder == null) {
                LOGGER.debug("no certificate present");
                return false;
            }
            Extensions extensions = certificateHolder.getExtensions();
            if (extensions == null) {
                LOGGER.debug("no extensions present");
                return false;
            }
            ExtendedKeyUsage extendedKeyUsage = ExtendedKeyUsage.fromExtensions((Extensions)extensions);
            if (extendedKeyUsage == null) {
                LOGGER.debug("no extension ExtendedKeyUsage present");
                return false;
            }
            if (!extendedKeyUsage.hasKeyPurposeId(KeyPurposeId.id_kp_timeStamping)) {
                LOGGER.debug("signer certificate not usable for timestamping");
                return false;
            }
            if (date != null && certificateHolder.getNotBefore().after(date) && date.after(certificateHolder.getNotAfter())) {
                LOGGER.debug("signer certificate not valid at time of signature creation");
                return false;
            }
            return true;
        }
    }
}

